// users.js

// Global variables
let users = [];
let roles = [];
let currentPage = 1;
let totalPages = 1;
let editingUserId = null;
const perPage = 10;

// API Configuration
const API_BASE_URL = window.location.origin; // Adjust this to your API base URL
const API_ENDPOINTS = {
    users: `${API_BASE_URL}/users`,
    roles: `${API_BASE_URL}/roles`
};

// DOM elements
const usersTableBody = document.getElementById('users-table-body');
const userModal = document.getElementById('user-modal');
const userForm = document.getElementById('user-form');
const modalTitle = document.getElementById('modal-title');
const submitBtn = document.getElementById('submit-btn');
const loadingIndicator = document.getElementById('loading-indicator');
const successNotification = document.getElementById('success-notification');
const errorNotification = document.getElementById('error-notification');

// API Helper functions
async function apiRequest(url, options = {}) {
    try {
        const config = {
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            },
            ...options
        };

        const response = await fetch(url, config);

        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.message || `HTTP error! status: ${response.status}`);
        }

        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            return await response.json();
        }

        return await response.text();
    } catch (error) {
        console.error('API request failed:', error);
        throw error;
    }
}

// API functions for data fetching
async function fetchUsers() {
    try {
        showLoading();
        const data = await apiRequest(API_ENDPOINTS.users);
        users = Array.isArray(data) ? data : (data.users || []);
        return users;
    } catch (error) {
        console.error('Failed to load users:', error);
        showNotification('error', 'Failed to load users: ' + error.message);
        // Fallback to empty array
        users = [];
        return users;
    } finally {
        hideLoading();
    }
}

async function fetchRoles() {
    try {
        const data = await apiRequest(API_ENDPOINTS.roles);
        roles = Array.isArray(data) ? data : (data.roles || []);
        return roles;
    } catch (error) {
        console.error('Failed to load roles:', error);
        showNotification('error', 'Failed to load roles: ' + error.message);
        // Fallback to empty array
        roles = [];
        return roles;
    }
}

async function createUser(userData) {
    return await apiRequest(API_ENDPOINTS.users, {
        method: 'POST',
        body: JSON.stringify(userData)
    });
}

async function updateUser(userId, userData) {
    return await apiRequest(`${API_ENDPOINTS.users}/${userId}`, {
        method: 'PUT',
        body: JSON.stringify(userData)
    });
}

async function deleteUserAPI(userId) {
    return await apiRequest(`${API_ENDPOINTS.users}/${userId}`, {
        method: 'DELETE'
    });
}

// Utility functions
function showLoading() {
    if (loadingIndicator) {
        loadingIndicator.style.display = 'flex';
    }
}

function hideLoading() {
    if (loadingIndicator) {
        loadingIndicator.style.display = 'none';
    }
}

function showNotification(type, message) {
    const notification = type === 'success' ? successNotification : errorNotification;
    if (!notification) return;

    const messageElement = document.getElementById(type + '-message');
    if (messageElement) {
        messageElement.textContent = message;
    }

    notification.style.display = 'flex';
    setTimeout(() => {
        notification.style.display = 'none';
    }, 4000);
}

function getAvatarColor(name) {
    const colors = ['#3b82f6', '#ef4444', '#10b981', '#f59e0b', '#8b5cf6', '#06b6d4', '#84cc16', '#f97316'];
    const index = name.charCodeAt(0) % colors.length;
    return colors[index];
}

function getInitials(name) {
    return (name.charAt(0)).toUpperCase();
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    try {
        return new Date(dateString).toLocaleDateString();
    } catch (error) {
        return 'Invalid Date';
    }
}

function formatDateTime(dateString) {
    if (!dateString) return 'Never';
    try {
        return new Date(dateString).toLocaleString();
    } catch (error) {
        return 'Invalid Date';
    }
}

// Table rendering
function renderUsersTable() {
    if (!usersTableBody) return;

    // Ensure users is an array
    if (!Array.isArray(users)) {
        users = [];
    }

    const filteredUsers = getFilteredUsers();
    const startIndex = (currentPage - 1) * perPage;
    const endIndex = startIndex + perPage;
    const paginatedUsers = filteredUsers.slice(startIndex, endIndex);

    usersTableBody.innerHTML = '';

    if (paginatedUsers.length === 0) {
        usersTableBody.innerHTML = `
            <tr>
                <td colspan="9" class="p-8 text-center text-gray-500">
                    <div class="flex flex-col items-center">
                        <svg class="w-12 h-12 mb-4 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197m13.5-9a2.5 2.5 0 11-5 0 2.5 2.5 0 015 0z"></path>
                        </svg>
                        <p class="text-lg font-medium">No users found</p>
                        <p class="text-sm text-gray-400 mt-1">
                            ${users.length === 0 ? 'No users have been created yet.' : 'Try adjusting your search or filter criteria.'}
                        </p>
                    </div>
                </td>
            </tr>
        `;
        return;
    }

    const editable = usersTableBody.classList.contains('editable');
    const deletable = usersTableBody.classList.contains('deletable');
    const importable = usersTableBody.classList.contains('importable');
    const exportable = usersTableBody.classList.contains('exportable');

    paginatedUsers.forEach(user => {
        if (!user) return;

        const row = document.createElement('tr');
        row.className = 'hover:bg-gray-50 transition-colors border-b border-gray-100';

        const avatarColor = getAvatarColor(user.name || 'User');
        const initials = getInitials(user.name || 'User');
        const fullName = user.name || 'Unknown User';

        // Get role display name
        const userRole = Array.isArray(roles) && user.roles && user.roles.length > 0
            ? roles.find(r => r && (r.id === user.roles[0].id || r.name === user.roles[0].name))
            : null;
        const roleDisplay = userRole ? userRole.name : (user.role || 'No Role');

        row.innerHTML = `
            <td class="p-4 border-r border-gray-100">
                <div class="user-avatar" style="background-color: ${avatarColor}">
                    ${initials}
                </div>
            </td>
            <td class="p-4 border-r border-gray-100">
                <div class="font-medium text-gray-900 capitalize">${fullName}</div>
            </td>
            <td class="p-4 border-r border-gray-100">
                <div class="text-gray-600">${user.email || 'No email'}</div>
            </td>
            <td class="p-4 border-r border-gray-100">
                <span class="role-badge role-${(userRole?.name || 'user').toLowerCase()}">${roleDisplay}</span>
            </td>
            <td class="p-4 border-r border-gray-100">
                <div class="text-sm text-gray-600">${formatDate(user.created_at)}</div>
            </td>
            <td class="p-4">
                <div class="flex space-x-2">
                    ${importable ? `<button onclick="importDataByUser(${user.id})" class="px-2 py-1 bg-green-600 text-white rounded hover:bg-green-700 transition-colors" title="Import Data">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"></path>
                        </svg>
                    </button>` : ''}
                    ${exportable ? `<button onclick="exportDataByUser(${user.id})" class="px-2 py-1 bg-indigo-600 text-white rounded hover:bg-indigo-700 transition-colors" title="Export Data">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-4-4m4 4l4-4m3 6H5a2 2 0 01-2-2V9a2 2 0 012-2h2l2-2h6l2 2h2a2 2 0 012 2v9a2 2 0 01-2 2z"></path>
                        </svg>
                    </button>` : ''}
                    ${editable ? `<button onclick="editUser(${user.id})" class="px-2 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 transition-colors" title="Edit User">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                        </svg>
                    </button>` : ''}
                    ${deletable ? `<button onclick="deleteUser(${user.id})" class="px-2 py-1 bg-red-600 text-white rounded hover:bg-red-700 transition-colors" title="Delete User">
                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                        </svg>
                    </button>` : ''}

                </div>
            </td>
        `;

        usersTableBody.appendChild(row);
    });

    updatePagination();
}

// Filtering and searching
function getFilteredUsers() {
    // Ensure users is an array
    if (!Array.isArray(users)) {
        return [];
    }

    let filtered = users.filter(user => user); // Remove any null/undefined users

    const searchInput = document.getElementById('search-input');
    const roleFilter = document.getElementById('role-filter');

    const searchTerm = searchInput ? searchInput.value.toLowerCase() : '';
    const roleFilterValue = roleFilter ? roleFilter.value : '';

    if (searchTerm) {
        filtered = filtered.filter(user => {
            const userRole = Array.isArray(roles) && user.roles && user.roles.length > 0
                ? roles.find(r => r && (r.id === user.roles[0].id || r.name === user.roles[0].name))
                : null;

            return (
                (user.name && user.name.toLowerCase().includes(searchTerm.toLowerCase())) ||
                (user.email && user.email.toLowerCase().includes(searchTerm.toLowerCase())) ||
                (userRole && userRole.name.toLowerCase().includes(searchTerm.toLowerCase()))
            );
        });
    }

    if (roleFilterValue) {
        filtered = filtered.filter(user => {
            const userRole = Array.isArray(roles) && user.roles && user.roles.length > 0
                ? roles.find(r => r && (r.id === user.roles[0].id || r.name === user.roles[0].name))
                : null;

            return userRole && (userRole.id == roleFilterValue || userRole.name === roleFilterValue);
        });
    }

    return filtered;
}

// Pagination
function updatePagination() {
    const filteredUsers = getFilteredUsers();
    totalPages = Math.ceil(filteredUsers.length / perPage);

    const showingFrom = Math.min((currentPage - 1) * perPage + 1, filteredUsers.length);
    const showingTo = Math.min(currentPage * perPage, filteredUsers.length);

    const showingFromEl = document.getElementById('showing-from');
    const showingToEl = document.getElementById('showing-to');
    const totalRecordsEl = document.getElementById('total-records');
    const currentPageEl = document.getElementById('current-page');
    const prevPageBtn = document.getElementById('prev-page');
    const nextPageBtn = document.getElementById('next-page');

    if (showingFromEl) showingFromEl.textContent = filteredUsers.length === 0 ? 0 : showingFrom;
    if (showingToEl) showingToEl.textContent = showingTo;
    if (totalRecordsEl) totalRecordsEl.textContent = filteredUsers.length;
    if (currentPageEl) currentPageEl.textContent = currentPage;

    if (prevPageBtn) prevPageBtn.disabled = currentPage <= 1;
    if (nextPageBtn) nextPageBtn.disabled = currentPage >= totalPages;
}

// Statistics update
function updateStatistics() {
    try {
        // Ensure users is an array
        if (!Array.isArray(users)) {
            users = [];
        }

        const totalUsers = users.length;
        const adminUsers = users.filter(u => {
            if (!u) return false;

            // Find the user's role object (from roles list) if possible
            const userRole = Array.isArray(roles) && u.roles && u.roles.length > 0
                ? roles.find(r => r && (r.id === u.roles[0].id || r.name === u.roles[0].name))
                : null;

            return (userRole && userRole.name.toLowerCase().includes('admin'));
        });

        const totalUsersEl = document.getElementById('total-users');
        const adminUsersEl = document.getElementById('admin-users');

        if (totalUsersEl) totalUsersEl.textContent = totalUsers.toLocaleString();
        if (adminUsersEl) adminUsersEl.textContent = adminUsers.length.toLocaleString();
    } catch (error) {
        console.error('Error updating statistics:', error);
    }
}

// Render roles dropdown
function renderRolesDropdown() {
    const roleSelect = document.getElementById('role');
    const roleFilter = document.getElementById('role-filter');

    if (roleSelect) {
        const currentValue = roleSelect.value;
        roleSelect.innerHTML = '<option value="">Select Role</option>';

        roles.forEach(role => {
            if (role && role.name) {
                const option = document.createElement('option');
                option.value = role.id;
                option.textContent = role.name;
                if (role.id == currentValue) {
                    option.selected = true;
                }
                roleSelect.appendChild(option);
            }
        });
    }

    if (roleFilter) {
        const currentFilterValue = roleFilter.value;
        roleFilter.innerHTML = '<option value="">All Roles</option>';

        roles.forEach(role => {
            if (role && role.name) {
                const option = document.createElement('option');
                option.value = role.id;
                option.textContent = role.name;
                if (role.id == currentFilterValue) {
                    option.selected = true;
                }
                roleFilter.appendChild(option);
            }
        });
    }
}

// Modal functions
function openModal() {
    if (userModal) {
        userModal.classList.add('active');
        document.body.style.overflow = 'hidden';
    }
}

function closeModal() {
    if (userModal) {
        userModal.classList.remove('active');
        document.body.style.overflow = 'auto';
        resetForm();
    }
}

function resetForm() {
    if (userForm) {
        userForm.reset();
    }
    editingUserId = null;
    if (modalTitle) modalTitle.textContent = 'Add New User';
    if (submitBtn) submitBtn.textContent = 'Create User';

    // Show password section for new users
    const passwordSection = document.getElementById('password-section');
    const passwordField = document.getElementById('password');
    const passwordConfirmField = document.getElementById('password-confirmation');

    if (passwordSection) passwordSection.style.display = 'block';
    if (passwordField) passwordField.required = true;
    if (passwordConfirmField) passwordConfirmField.required = true;
}

// User CRUD operations
function addUser() {
    resetForm();
    openModal();
}

function editUser(userId) {
    // Ensure users is an array
    if (!Array.isArray(users)) {
        users = [];
    }

    const user = users.find(u => u && u.id === userId);
    if (!user) return;

    const userRole = Array.isArray(roles) && user.roles && user.roles.length > 0
        ? roles.find(r => r && (r.id === user.roles[0].id || r.name === user.roles[0].name))
        : null;

    editingUserId = userId;
    if (modalTitle) modalTitle.textContent = 'Edit User';
    if (submitBtn) submitBtn.textContent = 'Update User';

    // Populate form fields
    const nameField = document.getElementById('name');
    const emailField = document.getElementById('email');
    const roleField = document.getElementById('role');

    if (nameField) nameField.value = user.name || '';
    if (emailField) emailField.value = user.email || '';
    if (roleField) roleField.value = userRole?.id || '';

    // Hide password section for editing
    const passwordSection = document.getElementById('password-section');
    const passwordField = document.getElementById('password');
    const passwordConfirmField = document.getElementById('password-confirmation');

    if (passwordSection) passwordSection.style.display = 'none';
    if (passwordField) passwordField.required = false;
    if (passwordConfirmField) passwordConfirmField.required = false;

    openModal();
}

function deleteUser(userId) {
    // Ensure users is an array
    if (!Array.isArray(users)) {
        users = [];
    }

    const user = users.find(u => u && u.id === userId);
    if (!user) return;

    const userName = user.first_name && user.last_name
        ? `${user.first_name} ${user.last_name}`
        : user.name || 'this user';

    if (confirm(`Are you sure you want to delete ${userName}? This action cannot be undone.`)) {
        performUserDelete(userId, user);
    }
}

function importDataByUser(userId) {
    document.getElementById('file-input').click();
    document.getElementById('user-input').value = userId;
}

if (document.getElementById('file-input')) {
    document.getElementById('file-input').addEventListener('change', async (e) => {

        const file = e.target.files[0];
        if (file) {
            const formData = new FormData();
            formData.append('file', file);
            formData.append('user_id', document.getElementById('user-input').value);

            const csrfToken = document.querySelector('meta[name="csrf-token"]').getAttribute('content');
            try {
                showLoading();
                const response = await fetch('/custom-data/import', {
                    method: 'POST',
                    body: formData,
                    headers: { 'X-CSRF-TOKEN': csrfToken }
                });

                const result = await response.json();

                if (response.ok && result.success) {
                    showNotification('success', result.message);
                    setTimeout(() => { location.reload(); }, 1500);
                } else {
                    throw new Error(result.message || 'Import failed');
                }
            } catch (error) {
                showNotification('error', 'Import failed: ' + error.message);
            } finally {
                hideLoading();
                e.target.value = '';
            }
        }
    });
}

function exportDataByUser(userId) {
    (async () => {
        try {
            const response = await apiRequest(`/custom-data/export/csv?user_id=${userId}`, 'GET');
            console.log(response);
            if (response.success) {
                const blob = new Blob([response.data], { type: 'text/csv' });
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `custom_data_export_${new Date().toISOString().split('T')[0]}.csv`;
                a.click();
                window.URL.revokeObjectURL(url);
                showNotification('success', 'Data exported successfully');
            }
        } catch (error) {
            showNotification('error', 'Failed to export data');
        }
    })();
}


async function performUserDelete(userId, user) {
    try {
        showLoading();
        await deleteUserAPI(userId);

        // Remove from local array
        users = users.filter(u => u && u.id !== userId);

        const userName = user.first_name && user.last_name
            ? `${user.first_name} ${user.last_name}`
            : user.name || 'User';

        showNotification('success', `${userName} has been deleted successfully`);
        renderUsersTable();
        updateStatistics();
    } catch (error) {
        showNotification('error', `Failed to delete user: ${error.message}`);
    } finally {
        hideLoading();
    }
}

// Form submission
async function handleFormSubmission(event) {
    event.preventDefault();

    const formData = new FormData(userForm);

    const userData = {
        first_name: formData.get('first_name')?.trim() || '',
        last_name: formData.get('last_name')?.trim() || '',
        name: formData.get('name')?.trim() || '',
        email: formData.get('email')?.trim() || '',
        phone: formData.get('phone')?.trim() || '',
        role_id: formData.get('role') || null,
        status: formData.get('status') || 'active',
        department: formData.get('department')?.trim() || '',
        notes: formData.get('notes')?.trim() || ''
    };

    // Add password fields for new users
    if (!editingUserId) {
        userData.password = formData.get('password');
        userData.password_confirmation = formData.get('password_confirmation');
    }

    // Validation
    if (!userData.email) {
        showNotification('error', 'Email address is required');
        return;
    }

    if (!userData.first_name && !userData.name) {
        showNotification('error', 'Name is required');
        return;
    }

    if (!editingUserId && (!userData.password || !userData.password_confirmation)) {
        showNotification('error', 'Password is required for new users');
        return;
    }

    if (!editingUserId && userData.password !== userData.password_confirmation) {
        showNotification('error', 'Passwords do not match');
        return;
    }

    // Check for duplicate email (except when editing)
    const existingUser = users.find(u =>
        u && u.email &&
        u.email.toLowerCase() === userData.email.toLowerCase() &&
        u.id !== editingUserId
    );

    if (existingUser) {
        showNotification('error', 'A user with this email address already exists');
        return;
    }

    try {
        showLoading();

        if (editingUserId) {
            // Update existing user
            const updatedUser = await updateUser(editingUserId, userData);
            const userIndex = users.findIndex(u => u && u.id === editingUserId);
            if (userIndex !== -1) {
                users[userIndex] = updatedUser;
            }
            showNotification('success', `User has been updated successfully`);
        } else {
            // Create new user
            const newUser = await createUser(userData);
            if (newUser) {
                users.push(newUser);
            }
            showNotification('success', `User has been created successfully`);
        }

        closeModal();
        renderUsersTable();
        updateStatistics();
    } catch (error) {
        showNotification('error', `Failed to ${editingUserId ? 'update' : 'create'} user: ${error.message}`);
    } finally {
        hideLoading();
    }
}

// Initialize application
async function initializeApp() {
    try {
        showLoading();

        // Load data from API
        await Promise.all([
            fetchRoles(),
            fetchUsers()
        ]);

        // Render initial UI
        renderRolesDropdown();
        renderUsersTable();
        updateStatistics();

    } catch (error) {
        console.error('Initialization error:', error);
        showNotification('error', 'Failed to initialize application: ' + error.message);

        // Ensure arrays are initialized even on error
        if (!Array.isArray(users)) users = [];
        if (!Array.isArray(roles)) roles = [];

        // Render UI with empty data
        renderRolesDropdown();
        renderUsersTable();
        updateStatistics();
    } finally {
        hideLoading();
    }
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    // Initialize the application
    initializeApp();

    // Button event listeners
    const addUserBtn = document.getElementById('add-user-btn');
    const closeModalBtn = document.getElementById('close-modal');
    const cancelBtn = document.getElementById('cancel-btn');

    if (addUserBtn) addUserBtn.addEventListener('click', addUser);
    if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
    if (cancelBtn) cancelBtn.addEventListener('click', closeModal);

    // Form submission
    if (userForm) {
        userForm.addEventListener('submit', handleFormSubmission);
    }

    // Search and filter functionality
    const searchInput = document.getElementById('search-input');
    const roleFilter = document.getElementById('role-filter');
    const statusFilter = document.getElementById('status-filter');

    if (searchInput) {
        searchInput.addEventListener('input', () => {
            currentPage = 1;
            renderUsersTable();
        });
    }

    if (roleFilter) {
        roleFilter.addEventListener('change', () => {
            currentPage = 1;
            renderUsersTable();
        });
    }

    if (statusFilter) {
        statusFilter.addEventListener('change', () => {
            currentPage = 1;
            renderUsersTable();
        });
    }

    // Pagination events
    const prevPageBtn = document.getElementById('prev-page');
    const nextPageBtn = document.getElementById('next-page');

    if (prevPageBtn) {
        prevPageBtn.addEventListener('click', () => {
            if (currentPage > 1) {
                currentPage--;
                renderUsersTable();
            }
        });
    }

    if (nextPageBtn) {
        nextPageBtn.addEventListener('click', () => {
            if (currentPage < totalPages) {
                currentPage++;
                renderUsersTable();
            }
        });
    }

    // Modal close on overlay click
    if (userModal) {
        userModal.addEventListener('click', function(event) {
            if (event.target === userModal) {
                closeModal();
            }
        });
    }

    // Close modal on Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape' && userModal?.classList.contains('active')) {
            closeModal();
        }
    });
});

// Global functions for onclick handlers
window.editUser = editUser;
window.deleteUser = deleteUser;
